﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/migrationhub-config/MigrationHubConfigRequest.h>
#include <aws/migrationhub-config/MigrationHubConfig_EXPORTS.h>
#include <aws/migrationhub-config/model/Target.h>

#include <utility>

namespace Aws {
namespace MigrationHubConfig {
namespace Model {

/**
 */
class CreateHomeRegionControlRequest : public MigrationHubConfigRequest {
 public:
  AWS_MIGRATIONHUBCONFIG_API CreateHomeRegionControlRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateHomeRegionControl"; }

  AWS_MIGRATIONHUBCONFIG_API Aws::String SerializePayload() const override;

  AWS_MIGRATIONHUBCONFIG_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the home region of the calling account.</p>
   */
  inline const Aws::String& GetHomeRegion() const { return m_homeRegion; }
  inline bool HomeRegionHasBeenSet() const { return m_homeRegionHasBeenSet; }
  template <typename HomeRegionT = Aws::String>
  void SetHomeRegion(HomeRegionT&& value) {
    m_homeRegionHasBeenSet = true;
    m_homeRegion = std::forward<HomeRegionT>(value);
  }
  template <typename HomeRegionT = Aws::String>
  CreateHomeRegionControlRequest& WithHomeRegion(HomeRegionT&& value) {
    SetHomeRegion(std::forward<HomeRegionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The account for which this command sets up a home region control. The
   * <code>Target</code> is always of type <code>ACCOUNT</code>.</p>
   */
  inline const Target& GetTarget() const { return m_target; }
  inline bool TargetHasBeenSet() const { return m_targetHasBeenSet; }
  template <typename TargetT = Target>
  void SetTarget(TargetT&& value) {
    m_targetHasBeenSet = true;
    m_target = std::forward<TargetT>(value);
  }
  template <typename TargetT = Target>
  CreateHomeRegionControlRequest& WithTarget(TargetT&& value) {
    SetTarget(std::forward<TargetT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional Boolean flag to indicate whether any effect should take place. It
   * tests whether the caller has permission to make the call.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline CreateHomeRegionControlRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_homeRegion;

  Target m_target;

  bool m_dryRun{false};
  bool m_homeRegionHasBeenSet = false;
  bool m_targetHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace MigrationHubConfig
}  // namespace Aws
